//	TorusGames3DTicTacToe.c
//
//	The board's n³ nodes are indexed in the obvious way
//	as (i,j,k), with i, j and k running in the x, y and z
//	directions, respectively.  The nodes sit wholly
//	within the game cell, so no node sits on any frame cell wall.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesMatrix44.h"
#include "GeometryGamesSound.h"
#include <math.h>
#include <float.h>


//	Size of markers as a fraction of a single tic-tac-toe cell.
//
//	Choose the cube's half-width to be
//
//		(1/2) * CubeRoot((4/3)π) * ball-radius
//
//	so that the cube and the ball have the same volume.
//
#define BALL_TO_CUBE_WIDTH_FACTOR	0.80599597700823482036
#define MARKER_BALL_RELATIVE_WIDTH	0.75	//	size of ball as fraction of enclosing cell
#define MARKER_CUBE_RELATIVE_WIDTH	(BALL_TO_CUBE_WIDTH_FACTOR * MARKER_BALL_RELATIVE_WIDTH)	//	size of cube as fraction of enclosing cell

//	Size of small marker (for unoccupied node) as fraction of
//	size of large marker (for occupied node).
#if TARGET_OS_IOS
#define SMALL_MARKER_FACTOR		0.50	//	50% size on handheld devices, where the user has a blunt finger
#else
#define SMALL_MARKER_FACTOR		0.25	//	25% size on desktop computers, where the user has a precise cursor
#endif


#define SELECTION_ANIMATION_DURATION_PART_1		0.25	//	seconds
#define SELECTION_ANIMATION_DURATION_PART_2		0.125	//	seconds
#define SELECTION_ANIMATION_DURATION_PART_3		0.125	//	seconds
#define WAIT_FOR_COMPUTER_TO_MOVE_DURATION		0.5		//	seconds

#define WIN_LINE_TUBE_RADIUS	0.046875


#if TIC_TAC_TOE_3D_SIZE == 3
#define TIC_TAC_TOE_MIDDLE	1	//	index of middle node
#else
#error Code assumes TIC_TAC_TOE_3D_SIZE == 3
#endif


//	Public functions with private names
static void			TicTacToeReset(ModelData *md);
static bool			TicTacToeDragBegin(ModelData *md, HitTestRay3D *aRay);
static void			TicTacToeDragObject(ModelData *md, HitTestRay3D *aRay, double aMotion[2]);
static void			TicTacToeDragEnd(ModelData *md, bool aTouchSequenceWasCancelled);
static unsigned int	TicTacToeGridSize(ModelData *md);
static void			TicTacToeSimulationUpdate(ModelData *md);

//	Private functions
static bool			RayHitsMarker(ModelData *md, HitTestRay3D *aRay, unsigned int aHitHode[3]);
static void			ComputerMoves(ModelData *md);
static bool			FindWinningMove(TicTacToePlayer aBoard[TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE],
						TopologyType aTopology, TicTacToePlayer aPlayer, unsigned int aWinningMove[3]);
static void			FindRandomMove(TicTacToePlayer aBoard[TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE],
						unsigned int aRandomMove[3]);
static bool			PlayerHasWon(TicTacToePlayer aBoard[TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE],
						TopologyType aTopology, TicTacToePlayer aPlayer, WinLine *aThreeInARow);
static void			NormalizeNode(signed int aNode[3], TopologyType aTopology);
static double		MarkerHalfWidth(TicTacToePlayer anActualOwner, TicTacToePlayer aPotentialOwner, bool aSelectedMarkerFlag);
static double		AnimatedMarkerHalfWidth(ModelData *md, TicTacToePlayer aPotentialOwner, bool aSelectedMarkerFlag);

static void			GetPolyhedronPlacementsForMarkers(ModelData *md, unsigned int aPlacementBufferLength, TorusGames3DPolyhedronPlacementAsCArrays *aPlacementBuffer);
static void			GetPolyhedronPlacementsForWinLine(ModelData *md, unsigned int aPlacementBufferLength, TorusGames3DPolyhedronPlacementAsCArrays *aPlacementBuffer);
static void			GetPolyhedronPlacementsForOneSliceThroughMarkers(
						ModelData *md, unsigned int anAxis, double anIntercept,
						double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
						unsigned int *aRectangularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aRectangularSlicePlacementBufferPtr,
						unsigned int *aCircularSlicePlacementBufferLengthPtr,			TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr);
static void			GetPolyhedronPlacementsForOneSliceThroughWinLineTube(
						ModelData *md, unsigned int anAxis, double anIntercept,
						double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
						unsigned int *aRectangularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aRectangularSlicePlacementBufferPtr,
						unsigned int *aCircularSlicePlacementBufferLengthPtr,			TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr,
						unsigned int *aClippedEllipticalSlicePlacementBufferLengthPtr,	TorusGames3DPolyhedronPlacementAsCArrays **aClippedEllipticalSlicePlacementBufferPtr);
static void			GetPolyhedronPlacementsForOneSliceThroughWinLineTube_Parallel(
						ModelData *md, unsigned int anAxis, double anIntercept,
						double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
						unsigned int *aRectangularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aRectangularSlicePlacementBufferPtr);
static void			GetPolyhedronPlacementsForOneSliceThroughWinLineTube_Transverse(
						ModelData *md, unsigned int anAxis, double anIntercept,
						double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
						unsigned int *aCircularSlicePlacementBufferLengthPtr,			TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr,
						unsigned int *aClippedEllipticalSlicePlacementBufferLengthPtr,	TorusGames3DPolyhedronPlacementAsCArrays **aClippedEllipticalSlicePlacementBufferPtr);
static void			GetPolyhedronPlacementsForOneSliceThroughWinLineEndcaps(
						ModelData *md, unsigned int anAxis, double anIntercept,
						double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
						unsigned int *aCircularSlicePlacementBufferLengthPtr,			TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr);
static double		GetMarkerHalfWidth(ModelData *md, unsigned int i, unsigned int j, unsigned int k, TicTacToePlayer *aPotentialOwner);


#ifdef __APPLE__
#pragma mark -
#pragma mark set up
#endif


void TicTacToe3DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &TicTacToeReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= NULL;
	md->itsGame2DDragObject				= NULL;
	md->itsGame2DDragEnd				= NULL;
	md->itsGame3DDragBegin				= &TicTacToeDragBegin;
	md->itsGame3DDragObject				= &TicTacToeDragObject;
	md->itsGame3DDragEnd				= &TicTacToeDragEnd;
	md->itsGame3DGridSize				= &TicTacToeGridSize;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= &TicTacToeSimulationUpdate;
	md->itsGameRefreshMessage			= NULL;
	
	//	Initialize tic-tac-toe variables.
	md->itsGameOf.TicTacToe3D.itsHitNode[0]			= 0;		//	unnecessary but safe
	md->itsGameOf.TicTacToe3D.itsHitNode[1]			= 0;
	md->itsGameOf.TicTacToe3D.itsHitNode[2]			= 0;
	md->itsGameOf.TicTacToe3D.itsHitRemainsValid	= false;	//	necessary
	md->itsGameOf.TicTacToe3D.itsHitIsByHuman		= false;	//	unnecessary but safe

	//	Set up the board.
	TicTacToeReset(md);
}


static void TicTacToeReset(ModelData *md)
{
	unsigned int	i,
					j,
					k;

	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
				md->itsGameOf.TicTacToe3D.itsBoard[i][j][k] = PlayerNone;

#ifdef MAKE_GAME_CHOICE_ICONS
	md->itsGameOf.TicTacToe3D.itsBoard[0][1][0] = PlayerX;
	md->itsGameOf.TicTacToe3D.itsBoard[1][2][1] = PlayerX;
	md->itsGameOf.TicTacToe3D.itsBoard[2][1][2] = PlayerX;

	md->itsGameOf.TicTacToe3D.itsBoard[0][2][2] = PlayerO;
	md->itsGameOf.TicTacToe3D.itsBoard[1][0][0] = PlayerO;
	md->itsGameOf.TicTacToe3D.itsBoard[2][0][1] = PlayerO;
#endif

	//	Abort any pending simulation.
	SimulationEnd(md);

	md->itsGameIsOver						= false;
	md->itsGameOf.TicTacToe3D.itsWhoseTurn	= PlayerX;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark drag
#endif


static bool TicTacToeDragBegin(
	ModelData		*md,
	HitTestRay3D	*aRay)
{
	md->itsGameOf.TicTacToe3D.itsHitRemainsValid =
	(
		RayHitsMarker(md, aRay, md->itsGameOf.TicTacToe3D.itsHitNode)
	 &&
		md->itsGameOf.TicTacToe3D.itsBoard	[md->itsGameOf.TicTacToe3D.itsHitNode[0]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[1]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[2]]
			== PlayerNone
	);

	if (md->itsGameOf.TicTacToe3D.itsHitRemainsValid)
	{
		//	Tentatively assign this cell to the player who tapped it.
		//	TicTacToeDragObject() may override this as the player
		//	slides his/her finger on or off the marker.
		//
		md->itsGameOf.TicTacToe3D.itsBoard	[md->itsGameOf.TicTacToe3D.itsHitNode[0]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[1]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[2]]
			= md->itsGameOf.TicTacToe3D.itsWhoseTurn;
	}
	
	md->itsGameOf.TicTacToe3D.itsHitIsByHuman = true;

	return md->itsGameOf.TicTacToe3D.itsHitRemainsValid;
}


static void TicTacToeDragObject(
	ModelData		*md,
	HitTestRay3D	*aRay,
	double			aMotion[2])
{
	unsigned int	theCurrentNode[3];

	UNUSED_PARAMETER(aMotion);
	
	if (RayHitsMarker(md, aRay, theCurrentNode)
	 && theCurrentNode[0] == md->itsGameOf.TicTacToe3D.itsHitNode[0]
	 && theCurrentNode[1] == md->itsGameOf.TicTacToe3D.itsHitNode[1]
	 && theCurrentNode[2] == md->itsGameOf.TicTacToe3D.itsHitNode[2])
	{
		md->itsGameOf.TicTacToe3D.itsHitRemainsValid = true;

		md->itsGameOf.TicTacToe3D.itsBoard	[md->itsGameOf.TicTacToe3D.itsHitNode[0]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[1]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[2]]
			= md->itsGameOf.TicTacToe3D.itsWhoseTurn;
	}
	else
	{
		md->itsGameOf.TicTacToe3D.itsHitRemainsValid = false;

		md->itsGameOf.TicTacToe3D.itsBoard	[md->itsGameOf.TicTacToe3D.itsHitNode[0]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[1]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[2]]
			= PlayerNone;
	}
}


static void TicTacToeDragEnd(
	ModelData	*md,
	bool		aTouchSequenceWasCancelled)
{
	//	Note:  If the user begins a trackpad gesture while
	//	a mouse motion is already in progress, or vice versa,
	//	the earlier motion might generate TicTacToeDragBegin()
	//	and zero or more TicTacToeDragObject() calls, but
	//	with no concluding TicTacToeDragEnd().
	//	This is harmless with the current code.
	//	Nothing bad happens, and the application won't crash.

	if (aTouchSequenceWasCancelled)	//	most commonly occurs when a gesture gets recognized
	{
		md->itsGameOf.TicTacToe3D.itsHitRemainsValid = false;

		md->itsGameOf.TicTacToe3D.itsBoard	[md->itsGameOf.TicTacToe3D.itsHitNode[0]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[1]]
											[md->itsGameOf.TicTacToe3D.itsHitNode[2]]
			= PlayerNone;
	}
	else
	{
		if (md->itsGameOf.TicTacToe3D.itsHitRemainsValid)
		{
			EnqueueSoundRequest(u"TicTacToeMove.mid");
			SimulationBegin(md, Simulation3DTicTacToeSelectNodePart1);
		}
		
		//	Wait until the end of Simulation3DTicTacToeSelectNodePart1
		//	to clear itsHitRemainsValid.
	}
}


static bool RayHitsMarker(
	ModelData		*md,			//	input
	HitTestRay3D	*aRay,			//	input
	unsigned int	aHitHode[3])	//	output: valid iff function returns true
{
	bool			theSuccessFlag;	//	= true iff aRay hits at least one marker,
									//		and the nearest such marker is for an unoccupied node
	double			theSmallestT;
	signed int		m = 0;	//	initialize to suppress compiler warnings
	unsigned int	i,
					j,
					k;
	TicTacToePlayer	theActualOwner,
					thePotentialOwner;
	bool			theMarkerIsSelected;
	double			theMarkerHalfWidth;	//	marker half-width as fraction of single tic-tac-toe cell size
	signed int		x,
					y,
					z;
	double			theMarkerCenterInGameCell[4],
					theMarkerCenterInTiling[4],
					theGameCellIntoTiling[4][4],
					theNewT;

	theSuccessFlag	= false;
	theSmallestT	= DBL_MAX;

	switch (md->itsViewType)
	{
		case ViewBasicLarge:

			//	Test for hits on the game cell's central image
			//	and its nearest neighbors -- that is, on the 3×3×3 grid
			//	of nearest images -- to guarantee that all visible images
			//	of each marker get tested.

			m = 1;	//	coordinates run -1 to +1

			break;
		
		case ViewRepeating:

			//	Test for hits over a larger grid, so the user
			//	may select images sitting deeper in the tiling.

			m = 2;	//	coordinates run -2 to +2 (anything deeper is in the fog)

			break;
		
		default:
			GeometryGamesFatalError(u"RayHitsMarker() received unexpected ViewType.", u"Internal Error");
			break;
	}
	
	theMarkerCenterInGameCell[3] = 1.0;

	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
	{
		theMarkerCenterInGameCell[0] = -0.5 + (0.5 + i)*TIC_TAC_TOE_3D_CELL_WIDTH;

		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
		{
			theMarkerCenterInGameCell[1] = -0.5 + (0.5 + j)*TIC_TAC_TOE_3D_CELL_WIDTH;

			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
			{
				theMarkerCenterInGameCell[2] = -0.5 + (0.5 + k)*TIC_TAC_TOE_3D_CELL_WIDTH;

				theActualOwner		= md->itsGameOf.TicTacToe3D.itsBoard[i][j][k];
				thePotentialOwner	= (theActualOwner != PlayerNone ?
										theActualOwner : md->itsGameOf.TicTacToe3D.itsWhoseTurn);
				theMarkerIsSelected	= (md->itsGameOf.TicTacToe3D.itsHitRemainsValid
									 && i == md->itsGameOf.TicTacToe3D.itsHitNode[0]
									 && j == md->itsGameOf.TicTacToe3D.itsHitNode[1]
									 && k == md->itsGameOf.TicTacToe3D.itsHitNode[2]);
				theMarkerHalfWidth	= MarkerHalfWidth(theActualOwner, thePotentialOwner, theMarkerIsSelected);

				for (x = -m; x <= +m; x++)
				{
					for (y = -m; y <= +m; y++)
					{
						for (z = -m; z <= +m; z++)
						{
							Make3DGameCellIntoTiling(theGameCellIntoTiling, x, y, z, md->itsTopology);
			
							Matrix44RowVectorTimesMatrix(	theMarkerCenterInGameCell,
															theGameCellIntoTiling,
															theMarkerCenterInTiling);
						
							if
							(
								(
									thePotentialOwner == PlayerX ?
									Ray3DIntersectsCube(	aRay,
															theMarkerCenterInTiling,
															theMarkerHalfWidth,
															&theNewT) :
									Ray3DIntersectsSphere(	aRay,
															theMarkerCenterInTiling,
															theMarkerHalfWidth,
															&theNewT)
								)
							 && theNewT > aRay->tMin
							 &&	theNewT < aRay->tMax
							 && theNewT < theSmallestT
							)
							{
								theSmallestT = theNewT;

								aHitHode[0] = i;
								aHitHode[1] = j;
								aHitHode[2] = k;

								theSuccessFlag = true;
							}
						}
					}
				}
			}
		}
	}
	
	return theSuccessFlag;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark game logic
#endif


static void ComputerMoves(ModelData *md)
{
	TicTacToePlayer	thePlayer,
					theOpponent;

	thePlayer	= md->itsGameOf.TicTacToe3D.itsWhoseTurn;
	theOpponent = (thePlayer == PlayerX ? PlayerO : PlayerX);

	if
	(
		//	try to win
		! FindWinningMove(
				md->itsGameOf.TicTacToe3D.itsBoard,
				md->itsTopology,
				thePlayer,
				md->itsGameOf.TicTacToe3D.itsHitNode)
	 &&
		//	try to block
		! FindWinningMove(
				md->itsGameOf.TicTacToe3D.itsBoard,
				md->itsTopology,
				theOpponent,
				md->itsGameOf.TicTacToe3D.itsHitNode)
	)
	{
		//	move randomly
		FindRandomMove(
				md->itsGameOf.TicTacToe3D.itsBoard,
				md->itsGameOf.TicTacToe3D.itsHitNode);
	}

	md->itsGameOf.TicTacToe3D.itsHitRemainsValid	= true;
	md->itsGameOf.TicTacToe3D.itsHitIsByHuman		= false;
	md->itsGameOf.TicTacToe3D.itsBoard	[md->itsGameOf.TicTacToe3D.itsHitNode[0]]
										[md->itsGameOf.TicTacToe3D.itsHitNode[1]]
										[md->itsGameOf.TicTacToe3D.itsHitNode[2]]
								= md->itsGameOf.TicTacToe3D.itsWhoseTurn;
}


static bool FindWinningMove(
	TicTacToePlayer	aBoard[TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE],	//	input
	TopologyType	aTopology,			//	input
	TicTacToePlayer	aPlayer,			//	input
	unsigned int	aWinningMove[3])	//	output, valid only when function returns true
{
	unsigned int	i,
					j,
					k;
	bool			theWinFlag;

	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
	{
		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
		{
			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
			{
				if (aBoard[i][j][k] == PlayerNone)
				{
					aBoard[i][j][k] = aPlayer;
					theWinFlag = PlayerHasWon(aBoard, aTopology, aPlayer, NULL);
					aBoard[i][j][k] = PlayerNone;
					
					if (theWinFlag)
					{
						aWinningMove[0] = i;
						aWinningMove[1] = j;
						aWinningMove[2] = k;
						return true;
					}
				}
			}
		}
	}

	return false;
}


static void FindRandomMove(
	TicTacToePlayer aBoard[TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE],
	unsigned int	aRandomMove[3])	//	output
{
	unsigned int	theNumEmptyNodes,
					i,
					j,
					k,
					theCount;

	//	Count the number of empty nodes.
	theNumEmptyNodes = 0;
	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
				if (aBoard[i][j][k] == PlayerNone)
					theNumEmptyNodes++;

	if (theNumEmptyNodes == 0)
		GeometryGamesFatalError(u"FindRandomMove() found theNumEmptyNodes == 0.", u"Internal Error");

	//	Move randomly.
	theCount = RandomUnsignedInteger() % theNumEmptyNodes;

	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
	{
		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
		{
			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
			{
				if (aBoard[i][j][k] == PlayerNone)
				{
					if (theCount-- == 0)
					{
						aRandomMove[0] = i;
						aRandomMove[1] = j;
						aRandomMove[2] = k;
						return;
					}
				}
			}
		}
	}

	GeometryGamesFatalError(u"FindRandomMove() found no empty node.", u"Internal Error");
}


static bool PlayerHasWon(
	TicTacToePlayer	aBoard[TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE],	//	input
	TopologyType	aTopology,			//	input
	TicTacToePlayer	aPlayer,			//	input, PlayerX or PlayerO
	WinLine			*aThreeInARow)		//	output, may be NULL
{
	signed int		theCenterNode[3],	//	Signed integers allow for ±1 offsets.
					theDelta[3],
					theNeighbor[3];
	
	//	Note:  PlayerHasWon() tests only for three-in-a-row,
	//	regardless of the TIC_TAC_TOE_3D_SIZE.

	//	Consider each possible center for the three-in-a-row.
	for (theCenterNode[0] = 0; theCenterNode[0] < TIC_TAC_TOE_3D_SIZE; theCenterNode[0]++)
	{
		for (theCenterNode[1] = 0; theCenterNode[1] < TIC_TAC_TOE_3D_SIZE; theCenterNode[1]++)
		{
			for (theCenterNode[2] = 0; theCenterNode[2] < TIC_TAC_TOE_3D_SIZE; theCenterNode[2]++)
			{
				//	Does aPlayer occupy theCenterNode?
				if (aBoard[theCenterNode[0]][theCenterNode[1]][theCenterNode[2]] != aPlayer)
					continue;

				//	Consider each possible direction.
				//	For each pair of opposite directions {theDelta, -theDelta},
				//	ignore one and examine the other.
				for (theDelta[0] = -1; theDelta[0] <= +1; theDelta[0]++)
				{
					if (theDelta[0] == -1)
						continue;
					
					for (theDelta[1] = -1; theDelta[1] <= +1; theDelta[1]++)
					{
						if (theDelta[0] == 0 && theDelta[1] == -1)
							continue;
						
						for (theDelta[2] = -1; theDelta[2] <= +1; theDelta[2]++)
						{
							if (theDelta[0] == 0 && theDelta[1] == 0 && theDelta[2] == -1)
								continue;
							if (theDelta[0] == 0 && theDelta[1] == 0 && theDelta[2] ==  0)
								continue;

							//	Examine the node in front of us in the given direction,
							//	and the node behind us in that direction.
							//	If both agree with theCenterNode, we've got a win.
							
							theNeighbor[0] = theCenterNode[0] + theDelta[0];
							theNeighbor[1] = theCenterNode[1] + theDelta[1];
							theNeighbor[2] = theCenterNode[2] + theDelta[2];
							NormalizeNode(theNeighbor,aTopology);
							if (aBoard[theNeighbor[0]][theNeighbor[1]][theNeighbor[2]] != aPlayer)
								continue;
							
							theNeighbor[0] = theCenterNode[0] - theDelta[0];
							theNeighbor[1] = theCenterNode[1] - theDelta[1];
							theNeighbor[2] = theCenterNode[2] - theDelta[2];
							NormalizeNode(theNeighbor, aTopology);
							if (aBoard[theNeighbor[0]][theNeighbor[1]][theNeighbor[2]] != aPlayer)
								continue;

							//	We've got a win!
							
							if (aThreeInARow != NULL)
							{
								aThreeInARow->itsCenterNode[0]	= theCenterNode[0];
								aThreeInARow->itsCenterNode[1]	= theCenterNode[1];
								aThreeInARow->itsCenterNode[2]	= theCenterNode[2];
								
								aThreeInARow->itsDirection[0]	= theDelta[0];
								aThreeInARow->itsDirection[1]	= theDelta[1];
								aThreeInARow->itsDirection[2]	= theDelta[2];
							}

							return true;
						}
					}
				}
			}
		}
	}

	//	Nobody's won yet.
	return false;
}


static void NormalizeNode(
	signed int		aNode[3],	//	each coordinate extends at most one unit beyond legal range
	TopologyType	aTopology)
{
	signed int	theSwapValue;

	//	Bring the x coordinate into the range {0, …, TIC_TAC_TOE_3D_SIZE - 1}.
	if (aNode[0] == TIC_TAC_TOE_3D_SIZE)
		aNode[0] = 0;
	if (aNode[0] == -1)
		aNode[0] = TIC_TAC_TOE_3D_SIZE - 1;

	//	Bring the y coordinate into the range {0, …, TIC_TAC_TOE_3D_SIZE - 1},
	//	accounting for the Klein space topology if required.
	if (aNode[1] == TIC_TAC_TOE_3D_SIZE
	 || aNode[1] == -1)
	{
		if (aNode[1] == TIC_TAC_TOE_3D_SIZE)
			aNode[1] = 0;
		else	//	aNode[1] == -1
			aNode[1] = TIC_TAC_TOE_3D_SIZE - 1;

		if (aTopology == Topology3DKlein)
			aNode[0] = (TIC_TAC_TOE_3D_SIZE - 1) - aNode[0];
	}

	//	Bring the z coordinate into the range {0, …, TIC_TAC_TOE_3D_SIZE - 1},
	//	accounting for the quarter-turn or half-turn topologies as required.
	if (aNode[2] == TIC_TAC_TOE_3D_SIZE
	 || aNode[2] == -1)
	{
		if (aNode[2] == TIC_TAC_TOE_3D_SIZE)
			aNode[2] = 0;
		else	//	aNode[2] == -1
			aNode[2] = TIC_TAC_TOE_3D_SIZE - 1;

		switch (aTopology)
		{
			case Topology3DQuarterTurn:

				//	Realize the quarter turn as the composition of two reflections.

				//	First reflect across the diagonal.
				theSwapValue	= aNode[0];
				aNode[0]		= aNode[1];
				aNode[1]		= theSwapValue;
				
				//	Then reflect vertically for a positive wrap
				//	or horizontally for a negative wrap.
				//
				//	Technical note:  For a positive wrap,
				//	the group rotates the fundamental cube
				//	by a 1/4 counterclockise turn when moving in the z direction.
				//	Here we're applying the inverse of that motion:
				//
				if (aNode[2] == 0)	//	wrapped z from TIC_TAC_TOE_3D_SIZE down to 0
				{
					//	Rotation is clockwise (when viewed by an observer
					//	looking in the positive z direction).
					aNode[1] = (TIC_TAC_TOE_3D_SIZE - 1) - aNode[1];
				}
				else				//	wrapped z from -1 up to TIC_TAC_TOE_3D_SIZE - 1
				{
					//	Rotation is counterclockwise (when viewed by an observer
					//	looking in the positive z direction).
					aNode[0] = (TIC_TAC_TOE_3D_SIZE - 1) - aNode[0];
				}

				break;
			
			case Topology3DHalfTurn:
			
				aNode[0] = (TIC_TAC_TOE_3D_SIZE - 1) - aNode[0];
				aNode[1] = (TIC_TAC_TOE_3D_SIZE - 1) - aNode[1];
				
				break;
			
			default:
				break;
		}
	}
}


#ifdef __APPLE__
#pragma mark -
#pragma mark grid
#endif


static unsigned int TicTacToeGridSize(ModelData *md)
{
	UNUSED_PARAMETER(md);

	return 3;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark simulation
#endif


static void TicTacToeSimulationUpdate(ModelData *md)
{
	double	s,
			t;

	switch (md->itsSimulationStatus)
	{
		case Simulation3DTicTacToeSelectNodePart1:

			if (md->itsSimulationElapsedTime > SELECTION_ANIMATION_DURATION_PART_1)
				md->itsSimulationElapsedTime = SELECTION_ANIMATION_DURATION_PART_1;

			t = md->itsSimulationElapsedTime / SELECTION_ANIMATION_DURATION_PART_1;
			s = 1.0 - t;

			if (md->itsGameOf.TicTacToe3D.itsHitIsByHuman)
			{
				md->itsGameOf.TicTacToe3D.itsSelectedMarkerSizeFactor	= s * SMALL_MARKER_FACTOR
																		+ t * 1.0;

				md->itsGameOf.TicTacToe3D.itsUnselectedMarkerSizeFactor	= s * SMALL_MARKER_FACTOR
																		+ t * 0.0;
			}
			else
			{
				md->itsGameOf.TicTacToe3D.itsSelectedMarkerSizeFactor	= s * 0.0
																		+ t * 1.0;

				md->itsGameOf.TicTacToe3D.itsUnselectedMarkerSizeFactor	= 0.0;
			}
			
			if (md->itsSimulationElapsedTime == SELECTION_ANIMATION_DURATION_PART_1)
			{
				SimulationEnd(md);

				md->itsGameOf.TicTacToe3D.itsHitRemainsValid = false;

				//	Has somebody won?
				if (PlayerHasWon(	md->itsGameOf.TicTacToe3D.itsBoard,
									md->itsTopology,
									md->itsGameOf.TicTacToe3D.itsWhoseTurn,
									&md->itsGameOf.TicTacToe3D.itsWinningThreeInARow))
				{
					md->itsGameIsOver = true;			//	Record the win.
					EnqueueSoundRequest(u"TicTacToeWin.mid");	//	Start a victory sound playing.
				}
				else
				{
					md->itsGameOf.TicTacToe3D.itsWhoseTurn =
						(md->itsGameOf.TicTacToe3D.itsWhoseTurn == PlayerX ? PlayerO : PlayerX);

					SimulationBegin(md, Simulation3DTicTacToeSelectNodePart2);
				}
			}

			break;

		case Simulation3DTicTacToeSelectNodePart2:

			if (md->itsSimulationElapsedTime >= SELECTION_ANIMATION_DURATION_PART_2)
			{
				SimulationEnd(md);
				
				//	Animate in the small markers for the new itsWhoseTurn.
				SimulationBegin(md, Simulation3DTicTacToeSelectNodePart3);
			}

			break;

		case Simulation3DTicTacToeSelectNodePart3:

			if (md->itsSimulationElapsedTime > SELECTION_ANIMATION_DURATION_PART_3)
				md->itsSimulationElapsedTime = SELECTION_ANIMATION_DURATION_PART_3;

			t = md->itsSimulationElapsedTime / SELECTION_ANIMATION_DURATION_PART_3;
			
			if (md->itsHumanVsComputer
			 && md->itsGameOf.TicTacToe3D.itsHitIsByHuman)
			{
				//	Don't expand the unselected markers for the computer's next move.
				md->itsGameOf.TicTacToe3D.itsMarkerSizeFactor	= 0.0;
			}
			else
			{
				//	Expand the unselected markers for a human's next move.
				md->itsGameOf.TicTacToe3D.itsMarkerSizeFactor	= t * SMALL_MARKER_FACTOR;
			}
			
			if (md->itsSimulationElapsedTime == SELECTION_ANIMATION_DURATION_PART_3)
			{
				SimulationEnd(md);

				if (md->itsHumanVsComputer
				 && md->itsGameOf.TicTacToe3D.itsHitIsByHuman)
				{
					//	Wait half a second before letting the computer respond,
					//	for the user's psychological benefit.
					SimulationBegin(md, Simulation3DTicTacToeWaitToMove);
				}
			}

			break;

		case Simulation3DTicTacToeWaitToMove:

			//	Wait half a second before letting the computer move.
			if (md->itsSimulationElapsedTime >= WAIT_FOR_COMPUTER_TO_MOVE_DURATION)
			{
				SimulationEnd(md);
				ComputerMoves(md);
				EnqueueSoundRequest(u"TicTacToeMove.mid");
				SimulationBegin(md, Simulation3DTicTacToeSelectNodePart1);
			}

			break;

		default:
			break;
	}
}


#ifdef __APPLE__
#pragma mark -
#pragma mark marker width
#endif

static double MarkerHalfWidth(
	TicTacToePlayer	anActualOwner,			//	PlayerX, PlayerO or PlayerNone
	TicTacToePlayer	aPotentialOwner,		//	PlayerX or PlayerO (agrees with anActualOwner when anActualOwner ≠ PlayerNone)
	bool			aSelectedMarkerFlag)	//	Is the player currently selecting this marker?
											//		If so, we still want to draw it small, even though we've assigned an owner.
{
	double	theRelativeWidth,
			theCellHalfWidth,
			theMarkerHalfWidth;	//	marker half-width as fraction of single tic-tac-toe cell size

	switch (aPotentialOwner)
	{
		case PlayerX:	theRelativeWidth = MARKER_CUBE_RELATIVE_WIDTH;	break;
		case PlayerO:	theRelativeWidth = MARKER_BALL_RELATIVE_WIDTH;	break;
		default:
			GeometryGamesFatalError(u"aPotentialOwner is invalid in MarkerHalfWidth() in TorusGames3DTicTacToe.c", u"Internal Error");
			theRelativeWidth = 0.0;	//	suppress compiler warnings
	}

	if (anActualOwner == PlayerNone
	 || aSelectedMarkerFlag)
	{
		theRelativeWidth *= SMALL_MARKER_FACTOR;
	}

	theCellHalfWidth	= 0.5 * TIC_TAC_TOE_3D_CELL_WIDTH;
	theMarkerHalfWidth	= theRelativeWidth * theCellHalfWidth;
	
	return theMarkerHalfWidth;
}

static double AnimatedMarkerHalfWidth(
	ModelData		*md,
	TicTacToePlayer	aPotentialOwner,
	bool			aSelectedMarkerFlag)
{
	double	theRelativeWidth,
			theCellHalfWidth,
			theMarkerHalfWidth;	//	marker half-width as fraction of single tic-tac-toe cell size

	switch (aPotentialOwner)
	{
		case PlayerX:	theRelativeWidth = MARKER_CUBE_RELATIVE_WIDTH;	break;
		case PlayerO:	theRelativeWidth = MARKER_BALL_RELATIVE_WIDTH;	break;
		default:
			GeometryGamesFatalError(u"aPotentialOwner is invalid in AnimatedMarkerHalfWidth() in TorusGames3DTicTacToe.c", u"Internal Error");
			theRelativeWidth = 0.0;	//	suppress compiler warnings
	}

	switch (md->itsSimulationStatus)
	{
		case Simulation3DTicTacToeSelectNodePart1:
			theRelativeWidth *= (aSelectedMarkerFlag ?
								md->itsGameOf.TicTacToe3D.itsSelectedMarkerSizeFactor :
								md->itsGameOf.TicTacToe3D.itsUnselectedMarkerSizeFactor);
			break;

		case Simulation3DTicTacToeSelectNodePart2:
			theRelativeWidth = 0.0;
			break;

		case Simulation3DTicTacToeSelectNodePart3:
			theRelativeWidth *= md->itsGameOf.TicTacToe3D.itsMarkerSizeFactor;
			break;
		
		case Simulation3DTicTacToeWaitToMove:
			theRelativeWidth = 0.0;
			break;
		
		default:
			GeometryGamesFatalError(u"Unexpected SimulationType in AnimatedMarkerHalfWidth()", u"Internal Error");
	}

	theCellHalfWidth	= 0.5 * TIC_TAC_TOE_3D_CELL_WIDTH;
	theMarkerHalfWidth	= theRelativeWidth * theCellHalfWidth;
	
	return theMarkerHalfWidth;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark polyhedron placements
#endif


unsigned int GetNum3DTicTacToePolyhedra_BufferPart1_Solids(
	ModelData	*md)
{
	UNUSED_PARAMETER(md);
	
	//	Get3DTicTacToePolyhedronPlacementsForSolids() returns the marker and win line placements
	//	in a well-defined order (see the nested loops in the code below)
	//	in the first part of the buffer.

	return TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE	//	game cells
			+ 1																//	(possibily unused) win line made of a tube...
			+ 2;															//		...and two endcaps
}

unsigned int GetNum3DTicTacToePolyhedra_BufferPart2_RectangularSlices(
	ModelData	*md)
{
	UNUSED_PARAMETER(md);

	//	The second part of the buffer is reserved for the "rectangular slices"
	//	where cubical markers and perhaps a win line intersect the frame cell boundary.
	//	Get3DTicTacToePolyhedronPlacementsForSolids() treats this second part of the buffer
	//	as a single homogeneous block, into which it writes as many
	//	slice placements as necessary, in no particular order.
	//	It fills the unused space in this part of the buffer with all-zero matrices,
	//	so that the rendering code will know which entries in this part
	//	of the buffer are in use and which are not.  More precisely:
	//
	//		if a placement matrix M   is   in use, then M[3][3] is guaranteed to be 1.0
	//		if a placement matrix M is not in use, then M[3][3] is guaranteed to be 0.0
	//
	
	return 3 *														//	rectangular slices; at most 3 frame cell faces will be visible,
		(															//		each frame cell face may intersect
			(TIC_TAC_TOE_3D_SIZE + 1) * (TIC_TAC_TOE_3D_SIZE + 1)	//		at most (n+1)² cubical marker images (in practice fewer) and
		  + 9														//		at most 9 face-parallel win line images (really at most 5, I think)
		);															//
}

unsigned int GetNum3DTicTacToePolyhedra_BufferPart3_CircularSlices(
	ModelData	*md)
{
	UNUSED_PARAMETER(md);

	//	The third part of the buffer is reserved for the "circular slices"
	//	where spherical markers and perhaps a win line intersect the frame cell boundary.
	//	This part of the buffer is treated as a single homogeneous block,
	//	as described in GetNum3DTicTacToePolyhedra_BufferPart2_RectangularSlices() above.

	return 3 *														//	circular slices; at most 3 frame cell faces will be visible,
		(															//		each frame cell face may intersect
			(TIC_TAC_TOE_3D_SIZE + 1) * (TIC_TAC_TOE_3D_SIZE + 1)	//		at most (n+1)² spherical marker images (in practice fewer),
		  + 8														//		at most 8 face-orthogonal win-line tube images (really fewer)
		  + 8														//		at most 8 win-line endcap images (really fewer)
		);
}

extern unsigned int	GetNum3DTicTacToePolyhedra_BufferPart4_ClippedEllipticalSlices(
	ModelData	*md)
{
	UNUSED_PARAMETER(md);

	//	The fourth part of the buffer is reserved for the (non-circular) "elliptical slices",
	//	where the frame call face slices an oblique (i.e. non-orthogonal) win line
	//	along a (non-circular) elliptical cross section, which may sometimes need
	//	additional clipping at either end of the tube.
	//
	//		To be clear:  All non-circular elliptical slices go here,
	//		whether they actually need the clipping or not.
	//		But circular slices go in buffer part 3 instead (see above).
	//
	//	This part of the buffer is treated as a single homogeneous block,
	//	as explained in GetNum3DPolyhedra_BufferPart2_RectangularSlices() above.

	return 3 *														//	circular slices; at most 3 frame cell faces will be visible,
		(															//		each frame cell face may intersect
			8														//		at most 8 oblique win-line tube images (really fewer)
		);
}


void Get3DTicTacToePolyhedronPlacementsForSolids(
	ModelData									*md,					//	input
	unsigned int								aPlacementBufferLength,	//	input (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aPlacementBuffer)		//	output buffer with space for aPlacementBufferLength placements
{
	unsigned int								theMarkerBufferLength,	//	in placements, not bytes
												theWinLineBufferLength;	//	in placements, not bytes
	TorusGames3DPolyhedronPlacementAsCArrays	*theMarkerBuffer,
												*theWinLineBuffer;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game3DTicTacToe,
		"Game3DTicTacToe must be active");

	theMarkerBufferLength	= TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE;
	theWinLineBufferLength	= 3;

	GEOMETRY_GAMES_ASSERT(
		aPlacementBufferLength == theMarkerBufferLength + theWinLineBufferLength,
		"Wrong buffer size");

	theMarkerBuffer		= aPlacementBuffer;
	theWinLineBuffer	= theMarkerBuffer + theMarkerBufferLength;

	GetPolyhedronPlacementsForMarkers(md, theMarkerBufferLength,  theMarkerBuffer );
	GetPolyhedronPlacementsForWinLine(md, theWinLineBufferLength, theWinLineBuffer);
}

static void GetPolyhedronPlacementsForMarkers(
	ModelData									*md,					//	input
	unsigned int								aPlacementBufferLength,	//	input, must equal n³ (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aPlacementBuffer)		//	output buffer with space for aPlacementBufferLength placements
{
	TorusGames3DPolyhedronPlacementAsCArrays	*thePlacement;
	unsigned int								i,
												j,
												k;
	double										theMarkerHalfWidth;	//	marker half-width as fraction of single tic-tac-toe cell size

	GEOMETRY_GAMES_ASSERT(
		aPlacementBufferLength == TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE,
		"Wrong buffer size");
	
	thePlacement = aPlacementBuffer;
	
	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
	{
		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
		{
			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
			{
				theMarkerHalfWidth = GetMarkerHalfWidth(md, i, j, k, NULL);
				
				thePlacement->itsDilation[0] = theMarkerHalfWidth;
				thePlacement->itsDilation[1] = theMarkerHalfWidth;
				thePlacement->itsDilation[2] = theMarkerHalfWidth;
				
				Matrix44Identity(thePlacement->itsIsometricPlacement);
				thePlacement->itsIsometricPlacement[3][0] = -0.5 + (0.5 + i)*TIC_TAC_TOE_3D_CELL_WIDTH;
				thePlacement->itsIsometricPlacement[3][1] = -0.5 + (0.5 + j)*TIC_TAC_TOE_3D_CELL_WIDTH;
				thePlacement->itsIsometricPlacement[3][2] = -0.5 + (0.5 + k)*TIC_TAC_TOE_3D_CELL_WIDTH;
				
				thePlacement->itsExtraClippingCovector[0] = 0.0;	//	unused for markers
				thePlacement->itsExtraClippingCovector[1] = 0.0;	//	unused for markers
				thePlacement->itsExtraClippingCovector[2] = 0.0;	//	unused for markers
				thePlacement->itsExtraClippingCovector[3] = 0.0;	//	unused for markers

				thePlacement++;
			}
		}
	}

	//	Just to be safe...
	GEOMETRY_GAMES_ASSERT(
		thePlacement - aPlacementBuffer == TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE * TIC_TAC_TOE_3D_SIZE,
		"Internal error:  wrong number of placements written to buffer");
}

static void GetPolyhedronPlacementsForWinLine(
	ModelData									*md,					//	input
	unsigned int								aPlacementBufferLength,	//	input, must equal 3 (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aPlacementBuffer)		//	output buffer with space for aPlacementBufferLength placements
{
	TorusGames3DPolyhedronPlacementAsCArrays	*thePlacement;
	WinLine										*theWinLine;
	double										theTubeLength,
												theTubeHalfLength,
												theTubeRadius,
												theCosAltitude,
												theSinAltitude,
												theAltitudeFactor[4][4],
												theAzimuthRadius,
												theCosAzimuth,
												theSinAzimuth,
												theAzimuthFactor[4][4],
												theTubePlacement[4][4],
												theShift[4][4];
	unsigned int								i,
												j;

	GEOMETRY_GAMES_ASSERT(
		aPlacementBufferLength == 3,
		"Wrong buffer size");

	thePlacement = aPlacementBuffer;

	if (md->itsGameIsOver)
	{
		theWinLine = &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow;
		
		theTubeLength		= sqrt( theWinLine->itsDirection[0] * theWinLine->itsDirection[0]
								  + theWinLine->itsDirection[1] * theWinLine->itsDirection[1]
								  + theWinLine->itsDirection[2] * theWinLine->itsDirection[2]);
		theTubeHalfLength	= 0.5 * theTubeLength;
		theTubeRadius		= WIN_LINE_TUBE_RADIUS;
		
		//	the tube

		//		dilational part of theTubePlacement
		//
		thePlacement->itsDilation[0] = theTubeHalfLength;
		thePlacement->itsDilation[1] = theTubeRadius;
		thePlacement->itsDilation[2] = theTubeRadius;

		//		rotational part of theTubePlacement
		//
		if (theWinLine->itsDirection[1] == 0	//	If itsDirection = (±1,0,0) ...
		 && theWinLine->itsDirection[2] == 0)
		{
			//	...the desired direction matches the tube's default direction,
			//	so the rotational part of theTubePlacement is the identity matrix.
			Matrix44Identity(theTubePlacement);
		}
		else	//	Otherwise...
		{
			//	...we need to rotate the tube to the desired direction.

			//	We could use a fancy spin-vector computation to compute theTubePlacement,
			//	but instead let's stick with a simple azimuth and altitude computation.
			
			//	Use a dot product to compute the altitude:
			//
			//		cos(altitude)	= (initial direction) · (desired direction)
			//						= (1, 0, 0) · (itsDirection / theTubeLength)
			//						= itsDirection[0] / theTubeLength
			//

			theCosAltitude = theWinLine->itsDirection[0] / theTubeLength;
			theSinAltitude = sqrt(1.0 - theCosAltitude*theCosAltitude);
			
			Matrix44Identity(theAltitudeFactor);
			theAltitudeFactor[0][0] =   theCosAltitude;
			theAltitudeFactor[0][1] =   theSinAltitude;
			theAltitudeFactor[1][0] = - theSinAltitude;
			theAltitudeFactor[1][1] =   theCosAltitude;
			
			//	Compute the azimuth directly.

			theAzimuthRadius = sqrt(theWinLine->itsDirection[1] * theWinLine->itsDirection[1]
								  + theWinLine->itsDirection[2] * theWinLine->itsDirection[2]);
			theCosAzimuth = theWinLine->itsDirection[1] / theAzimuthRadius;
			theSinAzimuth = theWinLine->itsDirection[2] / theAzimuthRadius;
			
			Matrix44Identity(theAzimuthFactor);
			theAzimuthFactor[1][1] =   theCosAzimuth;
			theAzimuthFactor[1][2] =   theSinAzimuth;
			theAzimuthFactor[2][1] = - theSinAzimuth;
			theAzimuthFactor[2][2] =   theCosAzimuth;
			
			Matrix44Product(theAltitudeFactor, theAzimuthFactor, theTubePlacement);
		}

		//		translational part of theTubePlacement
		for (i = 0; i < 3; i++)
			theTubePlacement[3][i] =  -0.5  +  (0.5 + theWinLine->itsCenterNode[i]) * TIC_TAC_TOE_3D_CELL_WIDTH;

		Matrix44Copy(thePlacement->itsIsometricPlacement, theTubePlacement);
		
		//		just to be tidy
		for (i = 0; i < 4; i++)
			thePlacement->itsExtraClippingCovector[i] = 0.0;	//	unused for win-line tube

		thePlacement++;

		//	the endcaps (if needed)
		if ( ! WinLineIsCircular(md->itsTopology, theWinLine) )
		{
			for (i = 0; i < 2; i++)
			{
				for (j = 0; j < 3; j++)
					thePlacement->itsDilation[j] = theTubeRadius;
				
				Matrix44Identity(theShift);
				theShift[3][0] = (i == 0 ? -theTubeHalfLength : +theTubeHalfLength);	//	shift the endcap by Δx = ±theTubeHalfLength
				Matrix44Product(theShift, theTubePlacement, thePlacement->itsIsometricPlacement);

				for (j = 0; j < 4; j++)
					thePlacement->itsExtraClippingCovector[j] = 0.0;	//	unused for win line endcaps

				thePlacement++;
			}
		}
		else
		{
			//	No endcaps are needed, so these values won't get used.
			//	Set some default values to create visible balls,
			//	so that if these values get used by accident, we'll notice immediately.
			for (i = 0; i < 2; i++)
			{
				for (j = 0; j < 3; j++)
					thePlacement->itsDilation[j] = 0.25;
				
				Matrix44Identity(thePlacement->itsIsometricPlacement);

				for (j = 0; j < 4; j++)
					thePlacement->itsExtraClippingCovector[j] = 0.0;	//	unused for win line endcaps
				
				thePlacement++;
			}
		}
	}
	else
	{
		//	No win line is needed, so these values won't get used.
		//	Set some default values to create a visible tube,
		//	so that if these values get used by accident, we'll notice immediately.
		for (i = 0; i < 3; i++)
		{
			for (j = 0; j < 3; j++)
				thePlacement->itsDilation[j] = 0.25;
			
			Matrix44Identity(thePlacement->itsIsometricPlacement);

			for (j = 0; j < 4; j++)
				thePlacement->itsExtraClippingCovector[j] = 0.0;	//	unused for win line endcaps

			thePlacement++;
		}
	}

	//	Just to be safe...
	GEOMETRY_GAMES_ASSERT(
		thePlacement - aPlacementBuffer == 3,
		"Internal error:  wrong number of placements written to buffer");
}


void Get3DTicTacToePolyhedronPlacementsForOneSlice(
	ModelData									*md,												//	input
	unsigned int								anAxis,												//	input
	double										anIntercept,										//	input
	double										aFrameCellCenterInGameCell[4],						//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],				//	input
	double										aGameCellIntoFrameCell[4][4],						//	input
	unsigned int								*aRectangularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aRectangularSlicePlacementBufferPtr,				//	input and output;  increment pointer after writing each placement
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr,					//	input and output;  increment pointer after writing each placement
	unsigned int								*aClippedEllipticalSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aClippedEllipticalSlicePlacementBufferPtr)		//	input and output;  increment pointer after writing each placement
{
	GEOMETRY_GAMES_ASSERT(md->itsGame == Game3DTicTacToe, "function called for wrong game");
	
	GetPolyhedronPlacementsForOneSliceThroughMarkers(
		md, anAxis, anIntercept,
		aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
		aRectangularSlicePlacementBufferLengthPtr,			aRectangularSlicePlacementBufferPtr,
		aCircularSlicePlacementBufferLengthPtr,				aCircularSlicePlacementBufferPtr);
	
	if (md->itsGameIsOver)
	{
		GetPolyhedronPlacementsForOneSliceThroughWinLineTube(
			md, anAxis, anIntercept,
			aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
			aRectangularSlicePlacementBufferLengthPtr,			aRectangularSlicePlacementBufferPtr,
			aCircularSlicePlacementBufferLengthPtr,				aCircularSlicePlacementBufferPtr,
			aClippedEllipticalSlicePlacementBufferLengthPtr,	aClippedEllipticalSlicePlacementBufferPtr);

		if ( ! WinLineIsCircular(md->itsTopology, &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow) )
		{
			GetPolyhedronPlacementsForOneSliceThroughWinLineEndcaps(
				md, anAxis, anIntercept,
				aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
				aCircularSlicePlacementBufferLengthPtr, aCircularSlicePlacementBufferPtr);
		}
	}
	else
	{
		//	The caller will zero out all unused polyhedron placements in the output buffers,
		//	so here we may simply ignore them.  This is in constrast to the placements
		//	for win-line tube and endcaps themselves, which GetPolyhedronPlacementsForWinLine()
		//	takes responsibility for setting to a reasonable default value when not in use.
	}
}

static void GetPolyhedronPlacementsForOneSliceThroughMarkers(
	ModelData									*md,												//	input
	unsigned int								anAxis,												//	input
	double										anIntercept,										//	input
	double										aFrameCellCenterInGameCell[4],						//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],				//	input
	double										aGameCellIntoFrameCell[4][4],						//	input
	unsigned int								*aRectangularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aRectangularSlicePlacementBufferPtr,				//	input and output;  increment pointer after writing each placement
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr)					//	input and output;  increment pointer after writing each placement
{
	double										theMarkerCenterInGameCell[4];
	unsigned int								i,
												j,
												k;
	double										theMarkerHalfWidth;	//	marker half-width as fraction of single tic-tac-toe cell size
	TicTacToePlayer								thePotentialOwner;
	bool										theMarkerMayBeCulled;
	TicTacToePlayer								theActualOwner;
	unsigned int								m;
	double										theSlicePlaneOffset,
												theSliceHalfWidth;
	TorusGames3DPolyhedronPlacementAsCArrays	*theSlicePlacement;


	//	All the slice placements for this slice will share the same rotational part,
	//	but with different dilational and translational parts.

	theMarkerCenterInGameCell[3] = 1.0;

	for (i = 0; i < TIC_TAC_TOE_3D_SIZE; i++)
	{
		theMarkerCenterInGameCell[0] = -0.5 + (0.5 + i)*TIC_TAC_TOE_3D_CELL_WIDTH;

		for (j = 0; j < TIC_TAC_TOE_3D_SIZE; j++)
		{
			theMarkerCenterInGameCell[1] = -0.5 + (0.5 + j)*TIC_TAC_TOE_3D_CELL_WIDTH;

			for (k = 0; k < TIC_TAC_TOE_3D_SIZE; k++)
			{
				theMarkerCenterInGameCell[2] = -0.5 + (0.5 + k)*TIC_TAC_TOE_3D_CELL_WIDTH;

				theMarkerHalfWidth = GetMarkerHalfWidth(md, i, j, k, &thePotentialOwner);
				
				//	If this marker lies wholly outside the frame cell, skip it.
				//
				//	This simple test will cull away
				//
				//		- all cubical markers that don't intersect the frame cell, and
				//
				//		- most spherical markers that don't intersect the frame cell.
				//
				//	Occasionally a spherical marker sitting outside the frame cell
				//	but near a (1-dimensional) frame cell edge may failed to get culled here,
				//	but that's OK:  such a spherical marker will get clipped away in the GPU.
				//	The purpose of this simple test is merely to avoid processing
				//	unneeded vertices in the GPU vertex function.  In other words,
				//	this test reduces the GPU's workload but isn't needed to insure correctness.
				//
				theMarkerMayBeCulled = false;
				for (m = 0; m < 3; m++)
				{
					 if (theMarkerCenterInGameCell[m] + theMarkerHalfWidth < aFrameCellCenterInGameCell[m] - 0.5)
					 	theMarkerMayBeCulled = true;

					 if (theMarkerCenterInGameCell[m] - theMarkerHalfWidth > aFrameCellCenterInGameCell[m] + 0.5)
					 	theMarkerMayBeCulled = true;
				}
				if (theMarkerMayBeCulled)
					continue;

				//	When the game is over, we don't draw markers at unoccupied nodes,
				//	so we shouldn't draw their cross sections either.
				theActualOwner = md->itsGameOf.TicTacToe3D.itsBoard[i][j][k];
				if (md->itsGameIsOver && theActualOwner == PlayerNone)
					continue;
				
				//	How far is the slice plane from the marker's center?
				theSlicePlaneOffset = anIntercept - theMarkerCenterInGameCell[anAxis];

				//	If this marker doesn't intersect the slice plane, skip it.
				if (theSlicePlaneOffset < -theMarkerHalfWidth
				 || theSlicePlaneOffset > +theMarkerHalfWidth)
				{
					continue;
				}
				
				//	Compute theSliceHalfWidth.
				switch (thePotentialOwner)
				{
					case PlayerNone:
						GEOMETRY_GAMES_ABORT("invalid potential owner");
						break;
					
					case PlayerX:	//	marker is a cube
						theSliceHalfWidth = theMarkerHalfWidth;
						break;
					
					case PlayerO:	//	marker is a sphere
						//	The above intersection test guarantees that
						//	the argument of the sqrt() will be non-negative
						//	(but it could be exactly zero -- there's no slack here!).
						theSliceHalfWidth = sqrt( theMarkerHalfWidth  * theMarkerHalfWidth
												- theSlicePlaneOffset * theSlicePlaneOffset);
						break;
				}

				//	Locate the next availble spot on the appropriate buffer
				//	to write the slice placement to.
				switch (thePotentialOwner)
				{
					case PlayerNone:
						GEOMETRY_GAMES_ABORT("invalid potential owner");
						break;
					
					case PlayerX:
						if (*aRectangularSlicePlacementBufferLengthPtr > 0)	//	should never fail
						{
							theSlicePlacement = *aRectangularSlicePlacementBufferPtr;
							(*aRectangularSlicePlacementBufferPtr)++;
							(*aRectangularSlicePlacementBufferLengthPtr)--;
						}
						else
						{
							theSlicePlacement = NULL;
						}
						break;
					
					case PlayerO:
						if (*aCircularSlicePlacementBufferLengthPtr > 0)	//	should never fail
						{
							theSlicePlacement = *aCircularSlicePlacementBufferPtr;
							(*aCircularSlicePlacementBufferPtr)++;
							(*aCircularSlicePlacementBufferLengthPtr)--;
						}
						else
						{
							theSlicePlacement = NULL;
						}
						break;
				}

				//	In principle we've ensured that enough buffer space will always be available.
				//	But in case it's not...
#ifdef DEBUG
				//	...during development we'd want to stop the app and resolve the problem, but...
				GEOMETRY_GAMES_ASSERT(
					theSlicePlacement != NULL,
					"internal error:  insufficient slice placement buffer space for marker slice");
#else
				//	...once the app is in the user's hands, we'd simply not draw this particular slice.
				if (theSlicePlacement == NULL)
					continue;
#endif
				
				//	Set the dilational part of the slice placement.
				//
				//		Note:  The dilational part gets applied in the slice's own local coordinate system.
				//		gSquareSliceVertices and gCircularSliceVertices place the square or disk in the xy plane,
				//		with its normal vector pointing in the negative z direction,
				//		so the dilations apply to the x and y coordinate, not the z coordinate.
				//
				theSlicePlacement->itsDilation[0] = theSliceHalfWidth;
				theSlicePlacement->itsDilation[1] = theSliceHalfWidth;
				theSlicePlacement->itsDilation[2] = 1.0;	//	safe but ultimately irrelevant,
															//		because the square and the disk both sit at z = 0
				
				//	Compute the isometric part of the slice placement in game cell coordinates
				//	(we'll convert it to frame cell coordinates immediately below).
				Matrix44Copy(theSlicePlacement->itsIsometricPlacement, aRotationalPartOfSlicePlacement);
				for (m = 0; m < 3; m++)
					theSlicePlacement->itsIsometricPlacement[3][m] = theMarkerCenterInGameCell[m];	//	value for m = anAxis gets overridden immediately below
				theSlicePlacement->itsIsometricPlacement[3][anAxis] = anIntercept;
				
				//	Convert the isometric placement from game cell to frame cell coordinates.
				Matrix44Product(	theSlicePlacement->itsIsometricPlacement,
									aGameCellIntoFrameCell,
									theSlicePlacement->itsIsometricPlacement);
				
				//	We won't need the extra clipping vector, so set it to zero.
				for (m = 0; m < 4; m++)
					theSlicePlacement->itsExtraClippingCovector[m] = 0.0;	//	unused for marker slices
			}
		}
	}
}

static void GetPolyhedronPlacementsForOneSliceThroughWinLineTube(
	ModelData									*md,												//	input
	unsigned int								anAxis,												//	input
	double										anIntercept,										//	input
	double										aFrameCellCenterInGameCell[4],						//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],				//	input
	double										aGameCellIntoFrameCell[4][4],						//	input
	unsigned int								*aRectangularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aRectangularSlicePlacementBufferPtr,				//	input and output;  increment pointer after writing each placement
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr,					//	input and output;  increment pointer after writing each placement
	unsigned int								*aClippedEllipticalSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aClippedEllipticalSlicePlacementBufferPtr)		//	input and output;  increment pointer after writing each placement
{
	WinLine	*theWinLine;

	theWinLine = &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow;
	
	GEOMETRY_GAMES_ASSERT(
			theWinLine->itsDirection[0] != 0
		 || theWinLine->itsDirection[1] != 0
		 || theWinLine->itsDirection[2] != 0,
		"internal error:  invalid win line (0,0,0)");

	if (theWinLine->itsDirection[anAxis] == 0)
	{
		//	The win line runs parallel to the frame cell wall.
		//	The intersection (if any) will be a rectangle.

		GetPolyhedronPlacementsForOneSliceThroughWinLineTube_Parallel(
			md, anAxis, anIntercept,
			aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
			aRectangularSlicePlacementBufferLengthPtr, aRectangularSlicePlacementBufferPtr);
	}
	else
	{
		//	The win line runs transverse to the frame cell wall.
		//	The intersection (if any) will be an ellipse.
		//	A non-circular ellipse may need to get clipped
		//	if it would otherwise extend beyond either end of the tube.
		
		GetPolyhedronPlacementsForOneSliceThroughWinLineTube_Transverse(
			md, anAxis, anIntercept,
			aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
			aCircularSlicePlacementBufferLengthPtr,				aCircularSlicePlacementBufferPtr,
			aClippedEllipticalSlicePlacementBufferLengthPtr,	aClippedEllipticalSlicePlacementBufferPtr);
	}
}

static void GetPolyhedronPlacementsForOneSliceThroughWinLineTube_Parallel(
	ModelData									*md,												//	input
	unsigned int								anAxis,												//	input
	double										anIntercept,										//	input
	double										aFrameCellCenterInGameCell[4],						//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],				//	input
	double										aGameCellIntoFrameCell[4][4],						//	input
	unsigned int								*aRectangularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aRectangularSlicePlacementBufferPtr)				//	input and output;  increment pointer after writing each placement
{
	WinLine										*theWinLine;
	double										theTubeCenter[3];
	signed int									theTubeDirection[3];	//	each component ∈ {-l, 0, +1}
	unsigned int								i,
												j,
												theTransverseAxes[2]	= {0, 0};	//	initialize to suppress compiler warnings
	double										theSlicePlaneOffset,
												theRectangleHalfLength,
												theRectangleHalfWidth,
												theSupplementaryRotation[4][4],
												theBoundingRectangleHalfSize[3],	//	only 2 of the components get used
												theDilationalPart[3],
												theIsometricPlacementInGameCell[4][4],
												theIsometricPlacementInFrameCell[4][4];
	TorusGames3DPolyhedronPlacementAsCArrays	*theSlicePlacement;

	GEOMETRY_GAMES_ASSERT(md->itsGameIsOver, "caller guarantees that game is over");
	
	theWinLine = &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow;
	
	for (i = 0; i < 3; i++)
	{
		theTubeCenter[i]	= -0.5  +  (0.5 + theWinLine->itsCenterNode[i]) * TIC_TAC_TOE_3D_CELL_WIDTH;
		theTubeDirection[i]	= theWinLine->itsDirection[i];
	}

	GEOMETRY_GAMES_ASSERT(theTubeDirection[anAxis] == 0, "caller guarantees that win-line tube is parallel to slice plane");
	
	theSlicePlaneOffset = anIntercept - theTubeCenter[anAxis];
	
	//	Does the tube intersect the slice plane?
	if (theSlicePlaneOffset >= -WIN_LINE_TUBE_RADIUS
	 && theSlicePlaneOffset <= +WIN_LINE_TUBE_RADIUS)
	{
		//	Compute theRectangleHalfWidth.
		//
		//		Note:  The above intersection test guarantees that
		//		the argument of the sqrt() will be non-negative
		//		(but it could be exactly zero -- there's no slack here!).
		//
		theRectangleHalfWidth = sqrt(WIN_LINE_TUBE_RADIUS * WIN_LINE_TUBE_RADIUS
									- theSlicePlaneOffset * theSlicePlaneOffset);

		//	The tube's intersection with the slicing plane will (if non-empty)
		//	be a long skinny rectangle, which we'll realize as the (highly anisotropic)
		//	image of a square.
		//
		//	The square's x direction (in the square's own local coordinate system)
		//	will map to the rectangle's long direction, while the square's y direction
		//	will map to the rectangle's short direction.  aRotationalPartOfSlicePlacement
		//	maps these two directions to theTransverseAxes[] in game coordinates.
		for (i = 0; i < 2; i++)
			for (j = 0; j < 3; j++)
				if (aRotationalPartOfSlicePlacement[i][j] != 0)
					theTransverseAxes[i] = j;
		
		GEOMETRY_GAMES_ASSERT(theTransverseAxes[0] != anAxis,				"internal inconsistency");
		GEOMETRY_GAMES_ASSERT(theTransverseAxes[1] != anAxis,				"internal inconsistency");
		GEOMETRY_GAMES_ASSERT(theTransverseAxes[0] != theTransverseAxes[1],	"internal inconsistency");

		switch (abs(theTubeDirection[theTransverseAxes[0]])
			  + abs(theTubeDirection[theTransverseAxes[1]]))
		{
			case 0:
				GEOMETRY_GAMES_ABORT("internal error:  theTubeDirection is (0,0,0)");
				break;
			
			case 1:

				//	The tube is parallel one of the transverse axes.
				
				theRectangleHalfLength = 0.5;
				
				if (theTubeDirection[theTransverseAxes[0]] != 0)
				{
					//	The square's x direction already gets mapped
					//	to the direction of the tube, so no further
					//	rotation is needed.
					Matrix44Identity(theSupplementaryRotation);

					theBoundingRectangleHalfSize[theTransverseAxes[0]] = theRectangleHalfLength;
					theBoundingRectangleHalfSize[theTransverseAxes[1]] = theRectangleHalfWidth;
				}
				else	//	theTubeDirection[theTransverseAxes[1]] != 0
				{
					//	The square's x direction gets mapped to a direction orthogonal to the tube,
					//	so we must rotate it the result ±90° to compensate (either direction is fine).
					Matrix44Identity(theSupplementaryRotation);
					theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[0]] =  0.0;
					theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[1]] = +1.0;
					theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[0]] = -1.0;
					theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[1]] =  0.0;

					theBoundingRectangleHalfSize[theTransverseAxes[0]] = theRectangleHalfWidth;
					theBoundingRectangleHalfSize[theTransverseAxes[1]] = theRectangleHalfLength;
				}
				
				break;
			
			case 2:

				//	The tube runs at a 45° angle relative to the transverse axes.
				
				theRectangleHalfLength = 0.5 * ROOT2;

				//	aRotationalPartOfSlicePlacement maps the square's x axis to theTransverseAxes[0],
				//	which we must rotate by an additional ±45° (choosing the direction correctly).
				if ((theTubeDirection[theTransverseAxes[0]] > 0)
				 == (theTubeDirection[theTransverseAxes[1]] > 0))
				{
					Matrix44Identity(theSupplementaryRotation);
					theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[0]] =  ROOT_HALF;
					theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[1]] = +ROOT_HALF;
					theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[0]] = -ROOT_HALF;
					theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[1]] =  ROOT_HALF;
				}
				else
				{
					Matrix44Identity(theSupplementaryRotation);
					theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[0]] =  ROOT_HALF;
					theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[1]] = -ROOT_HALF;
					theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[0]] = +ROOT_HALF;
					theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[1]] =  ROOT_HALF;
				}

				//	Because the rectangular cross section is tilted 45°, its bounding rectangle is a square.
				//	To compute the bounding square's half-height, start with the cross-sectional rectangle
				//	in standard position, with its corners at (±HL, ±HW), where HL denotes the half-length
				//	and HW denotes the half-width.  Rotating 45° counterclockwise gives
				//
				//			(±HL, ±HW) (  √½  √½ ) = √½ (±HL - ±HW, ±HL + ±HW)
				//			           ( -√½  √½ )
				//
				//	which shows that the bounding square's half-width and half-height are both
				//
				//			√½ ( HL + HW )
				//
				theBoundingRectangleHalfSize[theTransverseAxes[0]] = ROOT_HALF * (theRectangleHalfLength + theRectangleHalfWidth);
				theBoundingRectangleHalfSize[theTransverseAxes[1]] = ROOT_HALF * (theRectangleHalfLength + theRectangleHalfWidth);
				
				break;
			
			default:
				GEOMETRY_GAMES_ABORT("The components of itsDirection must all be -1, 0 or +1");
				break;
		}

		//	Cull this particular cross-sectional rectangle iff it lies wholly outside the frame cell
		//	in either of the two remaining dimensions.  In other words, accept this rectangle
		//	iff it intersects the frame cell in both of the two remaining dimensions.
		//	Note that this test will allow an occasional false positive to slip through,
		//	because a diagonally oriented rectangle may intersect the strip -1/2 ≤ x ≤ +1/2
		//	and also intersect the strip -1/2 ≤ y ≤ +1/2,
		//	without intersecting the square {(x,y) | -1/2 ≤ x ≤ +1/2 and -1/2 ≤ y ≤ +1/2}.
		//	But that's OK:  the GPU will clip away all such false positives.
		//	The culling that we're doing here reduces the GPU workload,
		//	but isn't needed to guarantee correctness.
		if
		(
				theTubeCenter[theTransverseAxes[0]] - theBoundingRectangleHalfSize[theTransverseAxes[0]]
			 <= aFrameCellCenterInGameCell[theTransverseAxes[0]] + 0.5
		 &&
		 		theTubeCenter[theTransverseAxes[0]] + theBoundingRectangleHalfSize[theTransverseAxes[0]]
			 >= aFrameCellCenterInGameCell[theTransverseAxes[0]] - 0.5
		 &&
				theTubeCenter[theTransverseAxes[1]] - theBoundingRectangleHalfSize[theTransverseAxes[1]]
			 <= aFrameCellCenterInGameCell[theTransverseAxes[1]] + 0.5
		 &&
		 		theTubeCenter[theTransverseAxes[1]] + theBoundingRectangleHalfSize[theTransverseAxes[1]]
			 >= aFrameCellCenterInGameCell[theTransverseAxes[1]] - 0.5
		)
		{
			//	Compute the dilational part of the slice placement.
			//
			//		Note:  The dilational part gets applied in the slice's own local coordinate system.
			//		gSquareSliceVertices places the square in the xy plane,
			//		with its normal vector pointing in the negative z direction,
			//		so the dilations apply to the x and y coordinate, not the z coordinate.
			//
			theDilationalPart[0] = theRectangleHalfLength;
			theDilationalPart[1] = theRectangleHalfWidth;
			theDilationalPart[2] = 1.0;	//	safe but ultimately irrelevant, because the square sits at z = 0

			//	Assemble the isometric part of the slice placement.

			//		First compose the full rotational part and then...
			Matrix44Product(aRotationalPartOfSlicePlacement, theSupplementaryRotation, theIsometricPlacementInGameCell);
			
			//		...add in the translational part and...
			for (i = 0; i < 3; i++)
				theIsometricPlacementInGameCell[3][i] = theTubeCenter[i];	//	start with the tube's center, then...
			theIsometricPlacementInGameCell[3][anAxis] = anIntercept;		//		...offset to the slice place

			//		...convert from game cell coordinates to frame cell coordinates.
			Matrix44Product(theIsometricPlacementInGameCell, aGameCellIntoFrameCell, theIsometricPlacementInFrameCell);

			//	Locate the next availble spot on the rectangular slice placement buffer.
			if (*aRectangularSlicePlacementBufferLengthPtr > 0)	//	should never fail
			{
				theSlicePlacement = *aRectangularSlicePlacementBufferPtr;
				(*aRectangularSlicePlacementBufferPtr)++;
				(*aRectangularSlicePlacementBufferLengthPtr)--;
			}
			else
			{
				//	In principle we've ensured that enough buffer space will always be available,
				//	so we should never get to this "else" clause.  But if we do, then...
#ifdef DEBUG
				//	...during development we'd want to stop the app and resolve the problem.
				GEOMETRY_GAMES_ABORT("internal error:  insufficient slice placement buffer space for win-line slice parallel to slicing plane");
#else
				//	...once the app is in the user's hands, we'd simply not draw this particular slice.
				return;
#endif
			}

			//	Copy the slice placement into the spot we just found.
			for (j = 0; j < 3; j++)
				theSlicePlacement->itsDilation[j] = theDilationalPart[j];
			Matrix44Copy(theSlicePlacement->itsIsometricPlacement, theIsometricPlacementInFrameCell);
			for (j = 0; j < 4; j++)
				theSlicePlacement->itsExtraClippingCovector[j] = 0.0;	//	unused for rectangular win-line cross sections
		}
	}
}

static void GetPolyhedronPlacementsForOneSliceThroughWinLineTube_Transverse(
	ModelData									*md,												//	input
	unsigned int								anAxis,												//	input
	double										anIntercept,										//	input
	double										aFrameCellCenterInGameCell[4],						//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],				//	input
	double										aGameCellIntoFrameCell[4][4],						//	input
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,			//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr,					//	input and output;  increment pointer after writing each placement
	unsigned int								*aClippedEllipticalSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aClippedEllipticalSlicePlacementBufferPtr)		//	input and output;  increment pointer after writing each placement
{
	WinLine										*theWinLine;
	double										theTubeCenter[3];
	signed int									theTubeDirection[3];	//	each component ∈ {-l, 0, +1}
	unsigned int								i,
												j,
												theTransverseAxes[2]	= {0, 0};	//	initialize to suppress compiler warnings
	double										theSliceMinorRadius,
												theSliceMajorRadius,
												theAspectRatio,
												theTubeHalfLengthSquared,
												theSupplementaryRotation[4][4],
												theBoundingRectangleHalfSize[3],	//	only 2 of the components get used
												theExtraExtension,
												t,
												theEllipseCenterInGameCell[3],
												theIsometricPlacementInGameCell[4][4],
												theIsometricPlacementInFrameCell[4][4];
	TorusGames3DPolyhedronPlacementAsCArrays	*theSlicePlacement;
	double										theExtraClippingCovectorInGameCell[4],	//	acts on a vector (x,y,z,1)
												theFrameCellIntoGameCell[4][4],
												theExtraClippingCovectorInFrameCell[4];

	GEOMETRY_GAMES_ASSERT(md->itsGameIsOver, "caller guarantees that game is over");

	theWinLine = &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow;
	
	for (i = 0; i < 3; i++)
	{
		theTubeCenter[i]	= -0.5  +  (0.5 + theWinLine->itsCenterNode[i]) * TIC_TAC_TOE_3D_CELL_WIDTH;
		theTubeDirection[i]	= theWinLine->itsDirection[i];
	}
	
	//	The original disk's x direction (in the disk's own local coordinate system)
	//	will serve as the ellipse's major axis, while the original disk's y direction
	//	will serve as the ellipse's minor axis.  aRotationalPartOfSlicePlacement
	//	maps these two directions to theTransverseAxes[] in game coordinates.
	for (i = 0; i < 2; i++)
		for (j = 0; j < 3; j++)
			if (aRotationalPartOfSlicePlacement[i][j] != 0)
				theTransverseAxes[i] = j;
	
	GEOMETRY_GAMES_ASSERT(theTransverseAxes[0] != anAxis,				"internal inconsistency");
	GEOMETRY_GAMES_ASSERT(theTransverseAxes[1] != anAxis,				"internal inconsistency");
	GEOMETRY_GAMES_ASSERT(theTransverseAxes[0] != theTransverseAxes[1],	"internal inconsistency");

	theSliceMinorRadius = WIN_LINE_TUBE_RADIUS;

	switch (abs(theTubeDirection[theTransverseAxes[0]])
		  + abs(theTubeDirection[theTransverseAxes[1]]))
	{
		case 0:

			//	The slice is a circle
			theAspectRatio				= 1.0;
			theSliceMajorRadius			= theAspectRatio * theSliceMinorRadius;
			theTubeHalfLengthSquared	= 0.25;

			Matrix44Identity(theSupplementaryRotation);

			theBoundingRectangleHalfSize[theTransverseAxes[0]] = theSliceMajorRadius;
			theBoundingRectangleHalfSize[theTransverseAxes[1]] = theSliceMinorRadius;

			break;
			
		case 1:

			//	The slice is an ellipse with aspect ratio √2:1
			theAspectRatio				= ROOT2;
			theSliceMajorRadius			= theAspectRatio * theSliceMinorRadius;
			theTubeHalfLengthSquared	= 0.50;
			
			if (theTubeDirection[theTransverseAxes[0]] != 0)
			{
				//	The ellipse is already oriented the way we want it.
				//	No supplementary rotation is needed.
				Matrix44Identity(theSupplementaryRotation);
				
				theBoundingRectangleHalfSize[theTransverseAxes[0]] = theSliceMajorRadius;
				theBoundingRectangleHalfSize[theTransverseAxes[1]] = theSliceMinorRadius;
			}
			else
			{
				//	To properly orient the ellipse, we must rotate it ±90° (either direction is fine).
				Matrix44Identity(theSupplementaryRotation);
				theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[0]] =  0.0;
				theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[1]] = +1.0;
				theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[0]] = -1.0;
				theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[1]] =  0.0;
				
				//	The bounding rectangle also gets rotated,
				//	which in effect swaps the major and minor axes.
				theBoundingRectangleHalfSize[theTransverseAxes[0]] = theSliceMinorRadius;
				theBoundingRectangleHalfSize[theTransverseAxes[1]] = theSliceMajorRadius;
			}

			break;
			
		case 2:

			//	The slice is an ellipse with aspect ratio √3:1
			theAspectRatio				= ROOT3;
			theSliceMajorRadius			= theAspectRatio * theSliceMinorRadius;
			theTubeHalfLengthSquared	= 0.75;

			//	To properly orient the ellipse, we must rotate it ±45° (choosing the direction correctly).
			if ((theTubeDirection[theTransverseAxes[0]] > 0)
			 == (theTubeDirection[theTransverseAxes[1]] > 0))
			{
				Matrix44Identity(theSupplementaryRotation);
				theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[0]] =  ROOT_HALF;
				theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[1]] = +ROOT_HALF;
				theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[0]] = -ROOT_HALF;
				theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[1]] =  ROOT_HALF;
			}
			else
			{
				Matrix44Identity(theSupplementaryRotation);
				theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[0]] =  ROOT_HALF;
				theSupplementaryRotation[theTransverseAxes[0]][theTransverseAxes[1]] = -ROOT_HALF;
				theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[0]] = +ROOT_HALF;
				theSupplementaryRotation[theTransverseAxes[1]][theTransverseAxes[1]] =  ROOT_HALF;
			}

			//	Because the ellipse is tilted 45°, its bounding rectangle is a square.
			//	The following computations shows that the bounding square's
			//	half width is √2 times the ellipse's minor radius.
			//
			//	Details:
			//
			//		For simplicity, start with an ellipse
			//		whose minor radius is 1 and whose major radius is √3.
			//		Align the coordinate system to the minor and major axes.
			//		The ellipse may be parameterized as
			//
			//			(x,y) = (cos θ, √3 sin θ)
			//
			//		and has derivative
			//
			//			(dx/dθ, dy/dθ) = (-sin θ, √3 cos θ).
			//
			//		The derivative points in the 45° direction when
			//
			//			sin θ = √3 cos θ
			//
			//		which occurs at
			//
			//			θ = π/3
			//
			//		For that value of θ we get
			//
			//			(x,y) = (1/2, 3/2)
			//
			//		That point lies on the 45° line
			//
			//			x + y = 2
			//
			//		That line comes closest to the origin at
			//
			//			(x,y) = (1,1)
			//
			//		So the minimal distance from that 45° line to the origin is √2.
			//
			theBoundingRectangleHalfSize[theTransverseAxes[0]] = ROOT2 * theSliceMinorRadius;
			theBoundingRectangleHalfSize[theTransverseAxes[1]] = ROOT2 * theSliceMinorRadius;

			break;
		
		default:
			GEOMETRY_GAMES_ABORT("The components of itsDirection must all be -1, 0 or +1");
			break;
	}

	//	Because the tube may run obliquely relative to the frame cell face,
	//	the tube's outer surface may extend slightly further than the tube's center line
	//	in the direction of the frame cell face.  A simple similar-triangles
	//	computation lets us deduce the magnitude of this extra extension.
	theExtraExtension = sqrt
						(
							( theAspectRatio * theAspectRatio  -  1.0 )
						  / (    theAspectRatio * theAspectRatio      )
						)
						* theSliceMinorRadius;

	//	Does the tube intersect the slice plane?
	if (theTubeCenter[anAxis] + 0.5 + theExtraExtension >= anIntercept
	 && theTubeCenter[anAxis] - 0.5 - theExtraExtension <= anIntercept)
	{
		//	Solve the equation
		//
		//		theTubeCenter[anAxis] + t * theTubeDirection[anAxis] = anIntercept
		//
		//	to find the value of t for which the win line passes through the clip plane.
		//
		//	Note:  The caller guarantees that theTubeDirection[anAxis] != 0,
		GEOMETRY_GAMES_ASSERT(
			theTubeDirection[anAxis] != 0,
			"Caller must ensure that theTubeDirection[anAxis] != 0");
		//	so there's no risk of division by zero here.
		//
		t = (anIntercept - theTubeCenter[anAxis]) / theTubeDirection[anAxis];
		
		//	Use t to compute the center the elliptical tube cross section, in game cell coordinates.
		for (i = 0; i < 3; i++)
			theEllipseCenterInGameCell[i] = theTubeCenter[i] + t * theTubeDirection[i];

		//	Cull this particular intersection disk iff it lies wholly outside the frame cell
		//	in either of the two remaining dimensions.  In other words, accept this intersection disk
		//	iff it intersects the frame cell in both of the two remaining dimensions.
		//	Note that this test will allow an occasional false positive to slip through,
		//	because a disk may intersect the strip -1/2 ≤ x ≤ +1/2
		//	and also intersect the strip -1/2 ≤ y ≤ +1/2,
		//	without intersecting the square {(x,y) | -1/2 ≤ x ≤ +1/2 and -1/2 ≤ y ≤ +1/2}.
		//	But that's OK:  the GPU will clip away all such false positives.
		//	The culling that we're doing here reduces the GPU workload,
		//	but isn't needed to guarantee correctness.
		if
		(
				theEllipseCenterInGameCell[theTransverseAxes[0]] - theBoundingRectangleHalfSize[theTransverseAxes[0]]
			 <= aFrameCellCenterInGameCell[theTransverseAxes[0]] + 0.5
		 &&
		 		theEllipseCenterInGameCell[theTransverseAxes[0]] + theBoundingRectangleHalfSize[theTransverseAxes[0]]
			 >= aFrameCellCenterInGameCell[theTransverseAxes[0]] - 0.5
		 &&
				theEllipseCenterInGameCell[theTransverseAxes[1]] - theBoundingRectangleHalfSize[theTransverseAxes[1]]
			 <= aFrameCellCenterInGameCell[theTransverseAxes[1]] + 0.5
		 &&
		 		theEllipseCenterInGameCell[theTransverseAxes[1]] + theBoundingRectangleHalfSize[theTransverseAxes[1]]
			 >= aFrameCellCenterInGameCell[theTransverseAxes[1]] - 0.5
		)
		{
			//	Assemble the isometric part of the slice placement.

			//		First compose the full rotational part and then...
			Matrix44Product(aRotationalPartOfSlicePlacement, theSupplementaryRotation, theIsometricPlacementInGameCell);
			
			//		...add in the translational part and...
			for (i = 0; i < 3; i++)
				theIsometricPlacementInGameCell[3][i] = theEllipseCenterInGameCell[i];

			//		...convert from game cell coordinates to frame cell coordinates.
			Matrix44Product(theIsometricPlacementInGameCell, aGameCellIntoFrameCell, theIsometricPlacementInFrameCell);
			

			//	A circular cross section will go to the circular slice placement buffer
			//		(and will get rendered with no extra clipping), but
			//
			//	a non-circular elliptical cross section will go to the clipped elliptical slice placement buffer
			//		(and will get clipped to respect the ends of the tube).
			//
			if (theAspectRatio == 1.0)	//	circular cross section
			{
				if (*aCircularSlicePlacementBufferLengthPtr > 0)			//	should never fail
				{
					theSlicePlacement = *aCircularSlicePlacementBufferPtr;
					(*aCircularSlicePlacementBufferPtr)++;
					(*aCircularSlicePlacementBufferLengthPtr)--;
				}
				else
				{
					theSlicePlacement = NULL;
				}
				
				for (i = 0; i < 4; i++)
					theExtraClippingCovectorInFrameCell[i] = 0.0;	//	value will get saved, but will be ignored at rendertime
			}
			else						//	(non-circular) elliptical cross section
			{
				if (*aClippedEllipticalSlicePlacementBufferLengthPtr > 0)	//	should never fail
				{
					theSlicePlacement = *aClippedEllipticalSlicePlacementBufferPtr;
					(*aClippedEllipticalSlicePlacementBufferPtr)++;
					(*aClippedEllipticalSlicePlacementBufferLengthPtr)--;
				}
				else
				{
					theSlicePlacement = NULL;
				}
				
				//	The GPU function will need to clip the elliptical cross section
				//	to keep it from extending beyond the ends of the tube.
				//	By convention, the GPU function will apply the covector to each vertex,
				//	and clip to the interval [-1.0, +1.0].
				//
				//	Let C = (C₀, C₁, C₂) be the win-line tube's centerpoint,
				//	and let P = (P₀, P₁, P₂) be an arbitrary point, in game cell coordinates.
				//	If v = (v₀, v₁, v₂) is covector "parallel to the tube", then the product
				//
				//		                            (v₀)
				//		(P₀ - C₀, P₁ - C₁, P₂ - C₂) (v₁)
				//		                            (v₂)
				//
				//	measures the distance along the tube's centerline.
				//	Furthermore, if we normalize the covector v to the inverse of the tube's half length,
				//	then the tube's endpoints will evaluate to -1 and +1.
				//
				for (i = 0; i < 3; i++)
					theExtraClippingCovectorInGameCell[i] = (theTubeDirection[i] * 0.5) / theTubeHalfLengthSquared;
				
				//	To fold the tube's centerpoint C into the covector, rewrite the above formula as
				//
				//		                (            v₀             )
				//		(P₀, P₁, P₂, 1) (            v₁             )
				//		                (            v₂             )
				//		                ( - (C₀ v₀ + C₁ v₁ + C₂ v₂) )
				//
				theExtraClippingCovectorInGameCell[3] = 0.0;
				for (i = 0; i < 3; i++)
					theExtraClippingCovectorInGameCell[3] -= theTubeCenter[i] * theExtraClippingCovectorInGameCell[i];

				//	We've just computed the extra clipping vector in game cell coordinates,
				//	but the GPU vertex function will want to evaluate it in frame cell coordinates
				//	(the reason:  for greater computational efficiency, the GPU vertex function
				//	maps each vertex from the object's home coordinates directly to frame cell coordinates,
				//	without wasting time to compute game cell coordinates along the way).
				//	To evaluate the extra clipping vector in game cell coordinates,
				//	conceptually we would
				//
				//		1.	start with a point q in frame cell coordinates,
				//
				//		2.	send the point from frame cell coordinates
				//			to game cell coordinates, and then
				//
				//		3.	evaluate extra clipping vector in game cell coordinates.
				//
				//	In matrix notation, that looks like
				//
				//		                ( frame cell ) (            v₀             )
				//		(q₀, q₁, q₂, 1) (     to     ) (            v₁             )
				//		                (  game cell ) (            v₂             )
				//		                (            ) ( - (C₀ v₀ + C₁ v₁ + C₂ v₂) )
				//
				//	Computationally it's simpler to premultiply the second and third factors in that product,
				//	that is, to precompute the product of the frame-cell-into-game-cell matrix
				//	with theExtraClippingCovectorInGameCell (which appears here as a column vector).
				//
				//		Note on terminology:  In more formal language, we're computing
				//		the "pullback" of the covector, from the game cell to the frame cell,
				//		but because we're using the frame-cell-into-game-cell mapping,
				//		which is already acting in the backwards direction,
				//		this "pullback" really feels more like a "pushfoward".
				//
				
				//	Invert aGameCellIntoFrameCell to get theFrameCellIntoGameCell.
				Matrix44GeometricInverse(aGameCellIntoFrameCell, theFrameCellIntoGameCell);
				
				//	Multiply theFrameCellIntoGameCell by the column vector theExtraClippingCovectorInGameCell
				//	to get theExtraClippingCovectorInFrameCell.
				Matrix44TimesColumnVector(	theFrameCellIntoGameCell,
											theExtraClippingCovectorInGameCell,
											theExtraClippingCovectorInFrameCell);
			}
			//	In principle we've ensured that enough buffer space will always be available.
			//	But in case it's not...
#ifdef DEBUG
			//	...during development we'd want to stop the app and resolve the problem, but...
			GEOMETRY_GAMES_ASSERT(
				theSlicePlacement != NULL,
				"internal error:  insufficient slice placement buffer space for marker slice");
#else
			//	...once the app is in the user's hands, we'd simply not draw this particular slice.
#endif

			if (theSlicePlacement != NULL)
			{
				//	Copy the slice placement into the spot we just found.

				theSlicePlacement->itsDilation[0] = theSliceMajorRadius;
				theSlicePlacement->itsDilation[1] = theSliceMinorRadius;
				theSlicePlacement->itsDilation[2] = 1.0;

				Matrix44Copy(theSlicePlacement->itsIsometricPlacement, theIsometricPlacementInFrameCell);
				
				for (i = 0; i < 4; i++)
					theSlicePlacement->itsExtraClippingCovector[i] = theExtraClippingCovectorInFrameCell[i];
			}
		}
	}
}

static void GetPolyhedronPlacementsForOneSliceThroughWinLineEndcaps(
	ModelData									*md,										//	input
	unsigned int								anAxis,										//	input
	double										anIntercept,								//	input
	double										aFrameCellCenterInGameCell[4],				//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],		//	input
	double										aGameCellIntoFrameCell[4][4],				//	input
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr)			//	input and output;  increment pointer after writing each placement
{
	WinLine										*theWinLine;
	unsigned int								i,
												j;
	double										theTubeCenter[3];		//	in game cell coordinates
	signed int									theTubeDirection[3];	//	each component ∈ {-l, 0, +1}
	double										theEndcapCenter[2][3],	//	in game cell coordinates
												theEndcapRadius;
	bool										theEndcapMayBeCulled;
	double										theSlicePlaneOffset,
												theSliceRadius,
												theDilationalPart[3],
												theTranslationalPart[4][4],
												theIsometricPlacementInGameCell[4][4],
												theIsometricPlacementInFrameCell[4][4];
	TorusGames3DPolyhedronPlacementAsCArrays	*theSlicePlacement;

	GEOMETRY_GAMES_ASSERT(md->itsGameIsOver, "caller guarantees that game is over");

	theWinLine = &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow;
	
	for (i = 0; i < 3; i++)
	{
		theTubeCenter[i]		= -0.5  +  (0.5 + theWinLine->itsCenterNode[i]) * TIC_TAC_TOE_3D_CELL_WIDTH;
		theTubeDirection[i]		= theWinLine->itsDirection[i];
		theEndcapCenter[0][i]	= theTubeCenter[i] - 0.5 * theTubeDirection[i];
		theEndcapCenter[1][i]	= theTubeCenter[i] + 0.5 * theTubeDirection[i];
	}
	theEndcapRadius = WIN_LINE_TUBE_RADIUS;

	for (i = 0; i < 2; i++)	//	which endcap
	{
		//	This simple test will cull away most (but not all) endcap images
		//	that don't intersect the frame cell.  Occasionally an endcap sitting
		//	outside the frame cell but near a (1-dimensional) frame cell edge
		//	may failed to get culled here, but that's OK:  such an endcap will
		//	get clipped away in the GPU.  The purpose of this simple test is
		//	merely to avoid processing unneeded vertices in the GPU vertex function.
		//	In other words, this test reduces the GPU's workload but isn't needed to insure correctness.
		//
		theEndcapMayBeCulled = false;
		for (j = 0; j < 3; j++)
		{
			 if (theEndcapCenter[i][j] + theEndcapRadius < aFrameCellCenterInGameCell[j] - 0.5)
				theEndcapMayBeCulled = true;

			 if (theEndcapCenter[i][j] - theEndcapRadius > aFrameCellCenterInGameCell[j] + 0.5)
				theEndcapMayBeCulled = true;
		}
		if (theEndcapMayBeCulled)
			continue;

		//	How far is the slice plane from the endcap's center?
		theSlicePlaneOffset = anIntercept - theEndcapCenter[i][anAxis];

		//	If this endcap doesn't intersect the slice plane, skip it.
		if (theSlicePlaneOffset < -theEndcapRadius
		 || theSlicePlaneOffset > +theEndcapRadius)
		{
			continue;
		}

		//	Compute theSliceRadius.
		//
		//		Note:  The above intersection test guarantees that
		//		the argument of the sqrt() will be non-negative
		//		(but it could be exactly zero -- there's no slack here!).
		//
		theSliceRadius = sqrt( theEndcapRadius  * theEndcapRadius
							 - theSlicePlaneOffset * theSlicePlaneOffset);

		//	Compute the dilational part of the slice placement.
		//
		//		Note:  The dilational part gets applied in the slice's own local coordinate system.
		//		gCircularSliceVertices places the disk in the xy plane,
		//		with its normal vector pointing in the negative z direction,
		//		so the dilations apply to the x and y coordinate, not the z coordinate.
		//
		theDilationalPart[0] = theSliceRadius;
		theDilationalPart[1] = theSliceRadius;
		theDilationalPart[2] = 1.0;	//	safe but ultimately irrelevant, because the disk sits at z = 0
				
		//	Compute the translational part of the slice placement.
		Matrix44Identity(theTranslationalPart);
		for (j = 0; j < 3; j++)
			theTranslationalPart[3][j] = theEndcapCenter[i][j];	//	value for j = anAxis gets overridden immediately below
		theTranslationalPart[3][anAxis] = anIntercept;
				
		//	Compose aRotationalPartOfSlicePlacement and theTranslationalPart, in that order,
		//	but don't include theDilationalPart.
		Matrix44Product(aRotationalPartOfSlicePlacement, theTranslationalPart, theIsometricPlacementInGameCell);
		Matrix44Product(theIsometricPlacementInGameCell, aGameCellIntoFrameCell, theIsometricPlacementInFrameCell);

		//	Locate the next availble spot on the circular slice placement buffer.
		if (*aCircularSlicePlacementBufferLengthPtr > 0)	//	should never fail
		{
			theSlicePlacement = *aCircularSlicePlacementBufferPtr;
			(*aCircularSlicePlacementBufferPtr)++;
			(*aCircularSlicePlacementBufferLengthPtr)--;
		}
		else
		{
			//	In principle we've ensured that enough buffer space will always be available,
			//	so we should never get to this "else" clause.  But if we do, then...
#ifdef DEBUG
			//	...during development we'd want to stop the app and resolve the problem.
			GEOMETRY_GAMES_ABORT("internal error:  insufficient slice placement buffer space for win-line endcap slice");
#else
			//	...once the app is in the user's hands, we'd simply not draw this particular slice.
			continue;
#endif
		}

		//	Copy the slice placement into the spot we just found.
		for (j = 0; j < 3; j++)
			theSlicePlacement->itsDilation[j] = theDilationalPart[j];
		Matrix44Copy(theSlicePlacement->itsIsometricPlacement, theIsometricPlacementInFrameCell);
		for (j = 0; j < 4; j++)
			theSlicePlacement->itsExtraClippingCovector[j] = 0.0;	//	unused for win-line endcap slices
	}
}


static double GetMarkerHalfWidth(	//	returns marker half-width as fraction of single tic-tac-toe cell size
	ModelData		*md,
	unsigned int	i,
	unsigned int	j,
	unsigned int	k,
	TicTacToePlayer	*aPotentialOwner)	//	may be NULL
{
	TicTacToePlayer	theActualOwner,
					thePotentialOwner;
	bool			theMarkerIsSelected;
	double			theMarkerHalfWidth;	//	marker half-width as fraction of single tic-tac-toe cell size

	theActualOwner = md->itsGameOf.TicTacToe3D.itsBoard[i][j][k];
	
	//	By convention, thePotentialOwner = theActualOwner if the cell is already taken,
	//	but other thePotentialOwner is whoever's turn it currently is.
	if (theActualOwner != PlayerNone)
		thePotentialOwner = theActualOwner;
	else
		thePotentialOwner = md->itsGameOf.TicTacToe3D.itsWhoseTurn;
	
	//	The caller might also want to know the potential owner.
	if (aPotentialOwner != NULL)
		*aPotentialOwner = thePotentialOwner;

	//	itsHitRemainsValid gets cleared at the end of Simulation3DTicTacToeSelectNodePart1,
	//	when the selected marker reaches its full size.  Until then, the selected marker
	//	gets drawn smaller.
	theMarkerIsSelected = (md->itsGameOf.TicTacToe3D.itsHitRemainsValid
						 && i == md->itsGameOf.TicTacToe3D.itsHitNode[0]
						 && j == md->itsGameOf.TicTacToe3D.itsHitNode[1]
						 && k == md->itsGameOf.TicTacToe3D.itsHitNode[2]);

	//	When the user taps a previously unowned marker, a series of animations begins, to:
	//
	//		1.	expand the newly tapped marker while shrinking other unowned markers
	//				(but not affecting already owned markers),
	//		2.	pause briefly, and then
	//		3.	re-expand the remaining unowned markers,
	//			but now with the opposite marker shape (cube or sphere).
	//
	if
	(
		(	md->itsSimulationStatus == Simulation3DTicTacToeSelectNodePart1	//	animation in progress?
		 || md->itsSimulationStatus == Simulation3DTicTacToeSelectNodePart2
		 || md->itsSimulationStatus == Simulation3DTicTacToeSelectNodePart3
		 || md->itsSimulationStatus == Simulation3DTicTacToeWaitToMove
		)
	 &&
		(
			theActualOwner == PlayerNone									//	unowned marker?
		 ||	theMarkerIsSelected												//	newly tapped marker
		)
	)
	{
		theMarkerHalfWidth = AnimatedMarkerHalfWidth(md, thePotentialOwner, theMarkerIsSelected);
	}
	else
	{
		theMarkerHalfWidth = MarkerHalfWidth(theActualOwner, thePotentialOwner, theMarkerIsSelected);
	}
	
	return theMarkerHalfWidth;
}


void Get3DTicTacToeAxisAlignedBoundingBox(
	ModelData	*md,									//	input
	double		someBoundingBoxCornersInGameCell[2][4])	//	output;  any pair of diametrically opposite corners
{
	WinLine			*theWinLine;
	unsigned int	i;

	//	Tentatively set the bounding to exactly match the game cell itself.
	//
	//		Design note:  To optimize performance we could
	//		inset the bounding box slightly, so that it's sure
	//		to include the largest possible markers without
	//		including all of the empty space around them.
	//		Indeed, we could optimize the bounding box further,
	//		to include only the markers currently in use.
	//		But to keep the code simple, let's not bother
	//		with any such optimization.
	//

	someBoundingBoxCornersInGameCell[0][0] = -0.5;
	someBoundingBoxCornersInGameCell[0][1] = -0.5;
	someBoundingBoxCornersInGameCell[0][2] = -0.5;
	someBoundingBoxCornersInGameCell[0][3] =  1.0;
	
	someBoundingBoxCornersInGameCell[1][0] = +0.5;
	someBoundingBoxCornersInGameCell[1][1] = +0.5;
	someBoundingBoxCornersInGameCell[1][2] = +0.5;
	someBoundingBoxCornersInGameCell[1][3] =  1.0;
	
	//	If a win line is present, its tube and perhaps also its endcaps
	//	may extend beyond the game cell.
	//
	if (md->itsGameIsOver)
	{
		theWinLine = &md->itsGameOf.TicTacToe3D.itsWinningThreeInARow;
		
		//	The following code assumes 3×3×3 Tic-Tac-Toe,
		//	but could be adapted for the general case if required.
		GEOMETRY_GAMES_ASSERT(TIC_TAC_TOE_3D_SIZE == 3, "bounding box assume 3x3x3 tic-tac-toe");

		//	For each coordinate direction ...
		for (i = 0; i < 3; i++)
		{
			//	Does the win line have a component in this direction?
			if (theWinLine->itsDirection[i] != 0)
			{
				//	To keep this code simple, allow for endcaps whether they're present or not.
				//	(Otherwise we'd need to deal with a possible non-axis-aligned tube,
				//	whose radius would push the bounding box out a little bit anyhow.)
				
				switch (theWinLine->itsCenterNode[i])
				{
					case 0:
						//	The win line extends beyond the game cell's
						//	negative face in this direction,
						//	but stays well clear of its positive face.
						someBoundingBoxCornersInGameCell[0][i] -= (TIC_TAC_TOE_3D_CELL_WIDTH + WIN_LINE_TUBE_RADIUS);
						break;

					case 1:
						//	The win line may extend beyond the game cell's
						//	negative and positive faces both in this direction,
						//	but only by the radius of the endcap.
						someBoundingBoxCornersInGameCell[0][i] -= WIN_LINE_TUBE_RADIUS;
						someBoundingBoxCornersInGameCell[1][i] += WIN_LINE_TUBE_RADIUS;
						break;

					case 2:
						//	The win line extends beyond the game cell's
						//	positive face in this direction,
						//	but stays well clear of its negative face.
						someBoundingBoxCornersInGameCell[1][i] += (TIC_TAC_TOE_3D_CELL_WIDTH + WIN_LINE_TUBE_RADIUS);
						break;
				}
			}
		}
	}
}

bool WinLineIsCircular(
	TopologyType	aTopology,
	WinLine			*aWinLine)
{
	switch (aTopology)
	{
		case Topology3DTorus:
			return true;
		
		case Topology3DKlein:
			return
				//	Win line doesn't cross the reversing face.
				aWinLine->itsDirection[1] == 0
			 ||
				//	Win line lies wholly within the central plane of glide symmetry.
				(	aWinLine->itsCenterNode[0] == TIC_TAC_TOE_MIDDLE
				 && aWinLine->itsDirection [0] == 0);
		
		case Topology3DQuarterTurn:
		case Topology3DHalfTurn:
			return
				//	Win line doesn't cross the rotating face.
				aWinLine->itsDirection[2] == 0
			 ||
				//	Win line runs along the corkscrew axis.
				(	aWinLine->itsCenterNode[0] == TIC_TAC_TOE_MIDDLE
				 && aWinLine->itsCenterNode[1] == TIC_TAC_TOE_MIDDLE
				 && aWinLine->itsDirection [0] == 0
				 && aWinLine->itsDirection [1] == 0);
		
		default:
			GeometryGamesFatalError(u"WinLineIsCircular() received a bad TopologyType.", u"Internal Error");
			return false;	//	suppresses compiler warnings
	}
}

